<?php
namespace Modules\Admin\Controllers;

use App\Controllers\BaseController;

class ManagementUser extends BaseController
{
    private function resolveNameColumn(string $table, array $candidates = ['nama','name','wilayah','group_wilayah']): string {
        try {
            $fields = db_connect()->getFieldNames($table);
            foreach ($candidates as $c) {
                if (in_array($c, $fields, true)) return $c;
            }
        } catch (\Throwable $e) {}
        // safest default
        return $candidates[0] ?? 'nama';
    }
    private function getRoleOptions(array $filters = []): array {
        try {
            $qb = db_connect()->table('role')->select('id, role')->orderBy('role','asc');
            if (!empty($filters['like'])) {
                $likes = (array)$filters['like'];
                $first = true;
                foreach ($likes as $lk) {
                    if ($first) { $qb->like('role', $lk); $first = false; }
                    else { $qb->orLike('role', $lk); }
                }
            }
            if (!empty($filters['whereIn'])) {
                foreach ($filters['whereIn'] as $col => $vals) { $qb->whereIn($col, (array)$vals); }
            }
            return $qb->get()->getResultArray();
        } catch (\Throwable $e) { return []; }
    }
    private function getLevelOptions(): array {
        $db = db_connect();
        $opt = ['l1'=>[], 'l2'=>[], 'l3'=>[]];
        try { $opt['l1'] = $db->table('level1')->select('id, level1')->orderBy('level1','asc')->get()->getResultArray(); } catch(\Throwable $e) {}
        try { $opt['l2'] = $db->table('level2')->select('id, level2')->orderBy('level2','asc')->get()->getResultArray(); } catch(\Throwable $e) {}
        try { $opt['l3'] = $db->table('level3')->select('id, level3')->orderBy('level3','asc')->get()->getResultArray(); } catch(\Throwable $e) {}
        return $opt;
    }

    public function index()
    {
        $db = db_connect();
        $rows = [];
        $gwCol = $this->resolveNameColumn('group_wilayah', ['nama','name','group_wilayah']);
        $wCol = $this->resolveNameColumn('wilayah', ['nama','name','wilayah']);
        try {
            $builder = $db->table('user u')
                ->select('u.id, u.nama, u.username, r.role, l1.level1, l2.level2, l3.level3')
                ->join('role r', 'r.id = u.role_id', 'left')
                ->join('level1 l1', 'l1.id = u.level1_id', 'left')
                ->join('level2 l2', 'l2.id = u.level2_id', 'left')
                ->join('level3 l3', 'l3.id = u.level3_id', 'left');

            // Conditionally join group_wilayah
            try {
                $userFields = $db->getFieldNames('user');
                $db->getFieldNames('group_wilayah');
                if (in_array('group_wilayah_id', $userFields, true)) {
                    $builder->select('gw.' . $gwCol . ' as group_wilayah_nama')->join('group_wilayah gw', 'gw.id = u.group_wilayah_id', 'left');
                } else {
                    $builder->select("'-' as group_wilayah_nama", false);
                }
            } catch (\Throwable $e) {
                $builder->select("'-' as group_wilayah_nama", false);
            }
            // Conditionally join wilayah
            try {
                $userFields = $userFields ?? $db->getFieldNames('user');
                $db->getFieldNames('wilayah');
                if (in_array('wilayah_id', $userFields, true)) {
                    // direct from user table
                    $builder->select('w.' . $wCol . ' as wilayah_nama')->join('wilayah w', 'w.id = u.wilayah_id', 'left');
                } else {
                    // derive from level2 if available
                    try {
                        $level2Fields = $db->getFieldNames('level2');
                        if (in_array('wilayah_id', $level2Fields, true)) {
                            $builder->select('w.' . $wCol . ' as wilayah_nama')->join('wilayah w', 'w.id = l2.wilayah_id', 'left');
                        } else {
                            $builder->select("'-' as wilayah_nama", false);
                        }
                    } catch (\Throwable $e2) {
                        $builder->select("'-' as wilayah_nama", false);
                    }
                }
            } catch (\Throwable $e) {
                $builder->select("'-' as wilayah_nama", false);
            }

            // Role-based scoping: admin_pusat can see all, BNNP/provinsi restricted to own wilayah
            $roleStr = strtolower((string)(session('role') ?? ''));
            if (strpos($roleStr, 'admin_pusat') === false && (strpos($roleStr, 'bnnp') !== false || strpos($roleStr, 'provinsi') !== false)) {
                // derive own wilayah via session level2
                try {
                    $sessionLevel2 = (int)(session('level2') ?? 0);
                    if ($sessionLevel2 > 0) {
                        $wcolL2 = 'wilayah_id';
                        $l2Fields = $db->getFieldNames('level2');
                        foreach (['wilayah_id','wilayah','id_wilayah'] as $c) { if (in_array($c, $l2Fields, true)) { $wcolL2 = $c; break; } }
                        $ownRow = $db->table('level2')->select($wcolL2.' as wilayah_id')->where('id',$sessionLevel2)->get()->getRowArray();
                        $ownWilayah = (int)($ownRow['wilayah_id'] ?? 0);
                        if ($ownWilayah > 0) {
                            $userFields2 = $userFields ?? $db->getFieldNames('user');
                            if (in_array('wilayah_id', $userFields2, true)) {
                                $builder->where('u.wilayah_id', $ownWilayah);
                            } else {
                                // fallback: filter via level2 join
                                $builder->where('l2.'.$wcolL2, $ownWilayah);
                            }
                        }
                    }
                } catch (\Throwable $e) { /* ignore scoping on error */ }
            }

            $rows = $builder->orderBy('u.id','desc')->get()->getResultArray();
        } catch (\Throwable $e) {
            $rows = [];
            try { log_message('error', 'ManagementUser index error: ' . $e->getMessage()); } catch (\Throwable $e2) {}
        }

        // Attach actions (icon-only)
        $rows = array_map(function($r){
            $id = (int)($r['id'] ?? 0);
            $l1 = trim((string)($r['level1'] ?? '')) !== '' ? (string)$r['level1'] : '-';
            $l2 = trim((string)($r['level2'] ?? '')) !== '' ? (string)$r['level2'] : '-';
            $l3 = trim((string)($r['level3'] ?? '')) !== '' ? (string)$r['level3'] : '-';
            // Compose single Level HTML with lines
            $r['level'] = 'Group : '.htmlspecialchars($l1, ENT_QUOTES, 'UTF-8')
                .'<br>Instansi : '.htmlspecialchars($l2, ENT_QUOTES, 'UTF-8')
                .'<br>Satuan Kerja : '.htmlspecialchars($l3, ENT_QUOTES, 'UTF-8');
            // Friendly group wilayah & wilayah with '-' fallback
            $gw = trim((string)($r['group_wilayah_nama'] ?? '')) !== '' ? (string)$r['group_wilayah_nama'] : '-';
            $wil = trim((string)($r['wilayah_nama'] ?? '')) !== '' ? (string)$r['wilayah_nama'] : '-';
            $r['group_wilayah'] = $gw;
            $r['wilayah'] = $wil;
            // Coalesce basic fields
            $r['id'] = $id;
            $r['nama'] = trim((string)($r['nama'] ?? '')) !== '' ? (string)$r['nama'] : '-';
            $r['username'] = trim((string)($r['username'] ?? '')) !== '' ? (string)$r['username'] : '-';
            $r['role'] = trim((string)($r['role'] ?? '')) !== '' ? (string)$r['role'] : '-';
            $actions = '<div class="inline-flex items-center justify-end gap-1">'
                .'<a data-modal class="p-1 rounded border bg-white hover:bg-slate-50" href="'.base_url('admin/user/show/'.$id).'" title="Detail" aria-label="Detail"><i data-lucide="eye" class="w-4 h-4"></i></a>'
                .'<a data-modal class="p-1 rounded border bg-white hover:bg-slate-50" href="'.base_url('admin/user/edit/'.$id).'" title="Edit" aria-label="Edit"><i data-lucide="pencil" class="w-4 h-4"></i></a>'
                .'<form method="post" action="'.base_url('admin/user/delete/'.$id).'" data-confirm="Yakin hapus user ini?" class="inline" title="Delete" aria-label="Delete">'.csrf_field().'<button type="submit" class="p-1 rounded border bg-white hover:bg-rose-50 text-rose-600"><i data-lucide="trash-2" class="w-4 h-4"></i></button></form>'
                .'<form method="post" action="'.base_url('admin/user/reset-password/'.$id).'" data-confirm="Reset password user ini?" class="inline" title="Reset Password" aria-label="Reset Password">'.csrf_field().'<button type="submit" class="p-1 rounded border bg-white hover:bg-sky-50 text-sky-600"><i data-lucide="key" class="w-4 h-4"></i></button></form>'
                .'</div>';
            $r['actions'] = $actions;
            // Optionally unset original level columns so table only shows the composed one
            unset($r['level1'], $r['level2'], $r['level3'], $r['group_wilayah_nama'], $r['wilayah_nama']);
            return $r;
        }, $rows);

        // Add sequential number for display (No Urut)
        $i = 1;
        foreach ($rows as &$r) { $r['no'] = $i++; }
        unset($r);

        return view('Modules\\Admin\\Views\\management_user', [
            'title' => 'Management User',
            'rows' => $rows,
        ]);
    }

    public function create()
    {
        $isModal = (string)($this->request->getGet('modal') ?? '') === '1' || $this->request->isAJAX();
        $levels = $this->getLevelOptions();
        $roleStr = strtolower((string)(session('role') ?? ''));
        // Default roles: all for admin_pusat, restricted for others below
        $roleOptions = $this->getRoleOptions();
        if (strpos($roleStr, 'kab/kota') !== false) {
            // Kab/Kota: limit Level1 to own, Level2 to own, Level3 under own; roles like 'kab/kota'
            try {
                $db = db_connect();
                $sessL1 = (int)(session('level1') ?? 0); $sessL2 = (int)(session('level2') ?? 0);
                if ($sessL1 > 0) { $levels['l1'] = $db->table('level1')->select('id, level1')->where('id',$sessL1)->get()->getResultArray(); }
                if ($sessL2 > 0) {
                    $levels['l2'] = $db->table('level2')->select('id, level2')->where('id',$sessL2)->get()->getResultArray();
                    $levels['l3'] = $db->table('level3')->select('id, level3')->where('level2_id',$sessL2)->orderBy('level3','asc')->get()->getResultArray();
                }
            } catch (\Throwable $e) {}
            $roleOptions = $this->getRoleOptions(['like' => ['kab/kota']]);
        } else if (strpos($roleStr, 'admin_pusat') === false && (strpos($roleStr, 'bnnp') !== false || strpos($roleStr, 'provinsi') !== false)) {
            try {
                $db = db_connect();
                $sessionLevel2 = (int)(session('level2') ?? 0);
                if ($sessionLevel2 > 0) {
                    $l2Fields = $db->getFieldNames('level2');
                    $wcol = 'wilayah_id';
                    foreach (['wilayah_id','wilayah','id_wilayah'] as $c) { if (in_array($c, $l2Fields, true)) { $wcol = $c; break; } }
                    $ownRow = $db->table('level2')->select($wcol.' as wilayah_id')->where('id',$sessionLevel2)->get()->getRowArray();
                    $ownWil = (int)($ownRow['wilayah_id'] ?? 0);
                    if ($ownWil > 0) {
                        try { $levels['l2'] = $db->table('level2')->select('id, level2')->where($wcol, $ownWil)->orderBy('level2','asc')->get()->getResultArray(); } catch (\Throwable $e) {}
                        try {
                            $levels['l3'] = $db->table('level3 l3')
                                ->select('l3.id, l3.level3')
                                ->join('level2 l2','l2.id=l3.level2_id','left')
                                ->where('l2.'.$wcol, $ownWil)
                                ->orderBy('l3.level3','asc')->get()->getResultArray();
                        } catch (\Throwable $e) {}
                    }
                }
            } catch (\Throwable $e) {}
            // Provinsi: roles include provinsi, kab/kota, user instansi pusat
            $roleOptions = $this->getRoleOptions(['like' => ['provinsi','kab/kota','user instansi pusat']]);
        }
        return view('Modules\\Admin\\Views\\management_user_form', [
            'title' => 'Tambah User',
            'action' => base_url('admin/user/store'),
            'method' => 'post',
            'value' => [ 'nama'=>'', 'username'=>'', 'role_id'=>'', 'level1_id'=>'', 'level2_id'=>'', 'level3_id'=>'', 'no_hp'=>'' ],
            'roles' => $roleOptions,
            'levels' => $levels,
            'submitLabel' => 'Simpan',
            'modal' => $isModal,
        ]);
    }

    public function store()
    {
        $req = $this->request;
        $nama = trim((string)$req->getPost('nama'));
        $username = trim((string)$req->getPost('username'));
        $password = (string)$req->getPost('password');
        if ($nama === '' || $username === '' || $password === '') {
            return redirect()->back()->with('error','Nama, Username, dan Password wajib diisi');
        }
        $data = [
            'nama' => $nama,
            'username' => $username,
            'password' => password_hash($password, PASSWORD_BCRYPT),
            'role_id' => (int)$req->getPost('role_id'),
            'level1_id' => (int)$req->getPost('level1_id'),
            'level2_id' => (int)$req->getPost('level2_id'),
            'level3_id' => (int)$req->getPost('level3_id'),
            'no_hp' => trim((string)$req->getPost('no_hp')),
        ];
        $roleStr = strtolower((string)(session('role') ?? ''));
        if (strpos($roleStr, 'admin_pusat') === false && (strpos($roleStr, 'bnnp') !== false || strpos($roleStr, 'provinsi') !== false)) {
            try {
                $db = db_connect();
                $sessionLevel2 = (int)(session('level2') ?? 0);
                if ($sessionLevel2 > 0) {
                    $l2Fields = $db->getFieldNames('level2');
                    $wcol = 'wilayah_id';
                    foreach (['wilayah_id','wilayah','id_wilayah'] as $c) { if (in_array($c, $l2Fields, true)) { $wcol = $c; break; } }
                    $ownRow = $db->table('level2')->select($wcol.' as wilayah_id')->where('id',$sessionLevel2)->get()->getRowArray();
                    $ownWil = (int)($ownRow['wilayah_id'] ?? 0);
                    if ($ownWil > 0) {
                        $l2id = (int)$data['level2_id'];
                        if ($l2id > 0) {
                            $ok = $db->table('level2')->where('id',$l2id)->where($wcol,$ownWil)->countAllResults() > 0;
                            if (!$ok) { return redirect()->back()->with('error','Tidak diperbolehkan memilih Instansi dari wilayah lain'); }
                        }
                        $l3id = (int)$data['level3_id'];
                        if ($l3id > 0) {
                            $ok3 = $db->table('level3 l3')->join('level2 l2','l2.id=l3.level2_id','left')->where('l3.id',$l3id)->where('l2.'.$wcol,$ownWil)->countAllResults() > 0;
                            if (!$ok3) { return redirect()->back()->with('error','Tidak diperbolehkan memilih Satuan Kerja dari wilayah lain'); }
                        }
                        $userFields = $db->getFieldNames('user');
                        if (in_array('wilayah_id', $userFields, true)) {
                            $wilPosted = (int)($req->getPost('wilayah_id') ?? 0);
                            if ($wilPosted === 0) { $data['wilayah_id'] = $ownWil; }
                            else if ($wilPosted !== $ownWil) { return redirect()->back()->with('error','Tidak diperbolehkan memilih wilayah lain'); }
                        }
                    }
                }
            } catch (\Throwable $e) {}
        }
        try { db_connect()->table('user')->insert($data); } catch (\Throwable $e) {
            return redirect()->back()->with('error','Gagal menyimpan user');
        }
        return redirect()->to(base_url('admin/user'))->with('success','User ditambahkan');
    }

    public function show(int $id)
    {
        $db = db_connect();
        $gwCol = $this->resolveNameColumn('group_wilayah', ['nama','name','group_wilayah']);
        $wCol = $this->resolveNameColumn('wilayah', ['nama','name','wilayah']);
        $builder = $db->table('user u')
            ->select('u.*, r.role, l1.level1, l2.level2, l3.level3')
            ->join('role r','r.id=u.role_id','left')
            ->join('level1 l1','l1.id=u.level1_id','left')
            ->join('level2 l2','l2.id=u.level2_id','left')
            ->join('level3 l3','l3.id=u.level3_id','left');
        try {
            $userFields = $db->getFieldNames('user');
            $db->getFieldNames('group_wilayah');
            if (in_array('group_wilayah_id', $userFields, true)) {
                $builder->select('gw.' . $gwCol . ' AS group_wilayah_nama')->join('group_wilayah gw','gw.id=u.group_wilayah_id','left');
            } else {
                $builder->select("'-' AS group_wilayah_nama", false);
            }
        } catch (\Throwable $e) {
            $builder->select("'-' AS group_wilayah_nama", false);
        }
        try {
            $userFields = $userFields ?? $db->getFieldNames('user');
            $db->getFieldNames('wilayah');
            if (in_array('wilayah_id', $userFields, true)) {
                $builder->select('w.' . $wCol . ' AS wilayah_nama')->join('wilayah w','w.id=u.wilayah_id','left');
            } else {
                // derive from level2 if available
                try {
                    $level2Fields = $db->getFieldNames('level2');
                    if (in_array('wilayah_id', $level2Fields, true)) {
                        $builder->select('w.' . $wCol . ' AS wilayah_nama')->join('wilayah w','w.id=l2.wilayah_id','left');
                    } else {
                        $builder->select("'-' AS wilayah_nama", false);
                    }
                } catch (\Throwable $e2) {
                    $builder->select("'-' AS wilayah_nama", false);
                }
            }
        } catch (\Throwable $e) {
            $builder->select("'-' AS wilayah_nama", false);
        }
        try {
            $row = $builder->where('u.id',$id)->get()->getRowArray();
        } catch (\Throwable $e) {
            try { log_message('error', 'ManagementUser show error: ' . $e->getMessage()); } catch (\Throwable $e2) {}
            $row = null;
        }
        if (!$row) return redirect()->to(base_url('admin/user'))->with('error','Data tidak ditemukan');
        // Compose friendly fields for view
        $l1 = trim((string)($row['level1'] ?? '')) !== '' ? (string)$row['level1'] : '-';
        $l2 = trim((string)($row['level2'] ?? '')) !== '' ? (string)$row['level2'] : '-';
        $l3 = trim((string)($row['level3'] ?? '')) !== '' ? (string)$row['level3'] : '-';
        $row['level'] = 'Group : '.$l1.'\nInstansi : '.$l2.'\nSatuan Kerja : '.$l3;
        $row['group_wilayah'] = trim((string)($row['group_wilayah_nama'] ?? '')) !== '' ? (string)$row['group_wilayah_nama'] : '-';
        $row['wilayah'] = trim((string)($row['wilayah_nama'] ?? '')) !== '' ? (string)$row['wilayah_nama'] : '-';
        unset($row['group_wilayah_nama'], $row['wilayah_nama']);
        $isModal = (string)($this->request->getGet('modal') ?? '') === '1' || $this->request->isAJAX();
        return view('Modules\\Admin\\Views\\management_user_show', [
            'title' => 'Detail User',
            'row' => $row,
            'modal' => $isModal,
        ]);
    }

    public function edit(int $id)
    {
        $db = db_connect();
        $row = $db->table('user')->where('id',$id)->get()->getRowArray();
        if (!$row) return redirect()->to(base_url('admin/user'))->with('error','Data tidak ditemukan');
        $isModal = (string)($this->request->getGet('modal') ?? '') === '1' || $this->request->isAJAX();
        $levels = $this->getLevelOptions();
        $roleStr = strtolower((string)(session('role') ?? ''));
        if (strpos($roleStr, 'admin_pusat') === false && (strpos($roleStr, 'bnnp') !== false || strpos($roleStr, 'provinsi') !== false)) {
            try {
                $db = db_connect();
                $sessionLevel2 = (int)(session('level2') ?? 0);
                if ($sessionLevel2 > 0) {
                    $l2Fields = $db->getFieldNames('level2');
                    $wcol = 'wilayah_id';
                    foreach (['wilayah_id','wilayah','id_wilayah'] as $c) { if (in_array($c, $l2Fields, true)) { $wcol = $c; break; } }
                    $ownRow = $db->table('level2')->select($wcol.' as wilayah_id')->where('id',$sessionLevel2)->get()->getRowArray();
                    $ownWil = (int)($ownRow['wilayah_id'] ?? 0);
                    if ($ownWil > 0) {
                        try { $levels['l2'] = $db->table('level2')->select('id, level2')->where($wcol, $ownWil)->orderBy('level2','asc')->get()->getResultArray(); } catch (\Throwable $e) {}
                        try {
                            $levels['l3'] = $db->table('level3 l3')
                                ->select('l3.id, l3.level3')
                                ->join('level2 l2','l2.id=l3.level2_id','left')
                                ->where('l2.'.$wcol, $ownWil)
                                ->orderBy('l3.level3','asc')
                                ->get()->getResultArray();
                        } catch (\Throwable $e) {}
                    }
                }
            } catch (\Throwable $e) {}
        }
        // Fallback roles list (unrestricted) if not set differently above
        $roleOptions = $this->getRoleOptions();
        return view('Modules\\Admin\\Views\\management_user_form', [
            'title' => 'Edit User',
            'action' => base_url('admin/user/update/'.$id),
            'method' => 'post',
            'value' => $row,
            'roles' => $roleOptions,
            'levels' => $levels,
            'submitLabel' => 'Update',
            'modal' => $isModal,
        ]);
    }

    public function update(int $id)
    {
        $req = $this->request;
        $nama = trim((string)$req->getPost('nama'));
        $username = trim((string)$req->getPost('username'));
        if ($nama === '' || $username === '') {
            return redirect()->back()->with('error','Nama dan Username wajib diisi');
        }
        $data = [
            'nama' => $nama,
            'username' => $username,
            'role_id' => (int)$req->getPost('role_id'),
            'level1_id' => (int)$req->getPost('level1_id'),
            'level2_id' => (int)$req->getPost('level2_id'),
            'level3_id' => (int)$req->getPost('level3_id'),
            'no_hp' => trim((string)$req->getPost('no_hp')),
        ];
        $roleStr = strtolower((string)(session('role') ?? ''));
        if (strpos($roleStr, 'admin_pusat') === false && (strpos($roleStr, 'bnnp') !== false || strpos($roleStr, 'provinsi') !== false)) {
            try {
                $db = db_connect();
                $sessionLevel2 = (int)(session('level2') ?? 0);
                if ($sessionLevel2 > 0) {
                    $l2Fields = $db->getFieldNames('level2');
                    $wcol = 'wilayah_id';
                    foreach (['wilayah_id','wilayah','id_wilayah'] as $c) { if (in_array($c, $l2Fields, true)) { $wcol = $c; break; } }
                    $ownRow = $db->table('level2')->select($wcol.' as wilayah_id')->where('id',$sessionLevel2)->get()->getRowArray();
                    $ownWil = (int)($ownRow['wilayah_id'] ?? 0);
                    if ($ownWil > 0) {
                        $l2id = (int)$req->getPost('level2_id');
                        if ($l2id > 0) {
                            $ok = $db->table('level2')->where('id',$l2id)->where($wcol,$ownWil)->countAllResults() > 0;
                            if (!$ok) { return redirect()->back()->with('error','Tidak diperbolehkan memilih Instansi dari wilayah lain'); }
                        }
                        $l3id = (int)$req->getPost('level3_id');
                        if ($l3id > 0) {
                            $ok3 = $db->table('level3 l3')->join('level2 l2','l2.id=l3.level2_id','left')->where('l3.id',$l3id)->where('l2.'.$wcol,$ownWil)->countAllResults() > 0;
                            if (!$ok3) { return redirect()->back()->with('error','Tidak diperbolehkan memilih Satuan Kerja dari wilayah lain'); }
                        }
                        $userFields = $db->getFieldNames('user');
                        if (in_array('wilayah_id', $userFields, true)) {
                            $wilPosted = (int)($req->getPost('wilayah_id') ?? 0);
                            if ($wilPosted === 0) { $data['wilayah_id'] = $ownWil; }
                            else if ($wilPosted !== $ownWil) { return redirect()->back()->with('error','Tidak diperbolehkan memilih wilayah lain'); }
                        }
                    }
                }
            } catch (\Throwable $e) {}
        }
        $password = (string)$req->getPost('password');
        if ($password !== '') { $data['password'] = password_hash($password, PASSWORD_BCRYPT); }
        try { db_connect()->table('user')->where('id',$id)->update($data); } catch (\Throwable $e) {
            return redirect()->back()->with('error','Gagal memperbarui user');
        }
        return redirect()->to(base_url('admin/user'))->with('success','User diperbarui');
    }

    public function delete(int $id)
    {
        try { db_connect()->table('user')->where('id',$id)->delete(); } catch (\Throwable $e) {
            return redirect()->to(base_url('admin/user'))->with('error','Gagal menghapus user');
        }
        return redirect()->to(base_url('admin/user'))->with('success','User dihapus');
    }

    public function resetPassword(int $id)
    {
        $newPass = '123456';
        try { db_connect()->table('user')->where('id',$id)->update(['password' => password_hash($newPass, PASSWORD_BCRYPT)]); }
        catch (\Throwable $e) { return redirect()->to(base_url('admin/user'))->with('error','Gagal reset password'); }
        return redirect()->to(base_url('admin/user'))->with('success','Password direset ke: '.$newPass);
    }
}
